//=============================================================================
// MYTH_CGC_BlockGenerate
//=============================================================================
/*:
* @target MZ
* @author Swarnava Banerjee (Neel)
* @plugindesc v1.3.1 Build Block on an Actor in the style of Slay the Spire
* @url https://mythatelier.itch.io/card-game-combat
*
* @============================================================================
* Parameters
* ============================================================================
*
* @param blockStatesList
* @text Block States
* @type struct<BlockState>[]
* @desc List of States which are used to Block Damage. List ID may be used for Notetags.
*
* @param multiBlockRule
* @parent blockStatesList
* @text Multi Block Rule
* @type select
* @option Ascending
* @option Descending
* @option Even Spread
* @desc Set rule for how DMG is dealt when multiple valid Block States are on a target. See Plugin Help File for details.
* @default Ascending
*
* @param allowSlipDMG
* @parent blockStatesList
* @text Allow Slip DMG/HP Degen?
* @desc If ON, Slip DMG/HP Degen will reduce a battler's Block before HP. If OFF, Slip DMG/HP Degen ignores Block and reduces HP.
* @type boolean
* @default false
*
* @param removeStateOnZero
* @parent blockStatesList
* @text Remove State on 0?
* @desc If ON, Block States will be removed if you have 0 Block for that State. If OFF, State will persist regardless of value.
* @type boolean
* @default false
*
* @param ----------------------
*
* @param rescStatesList
* @text Resource States
* @type struct<RescState>[]
* @desc List of States which are used to track Resources like HP, MP or TP. List ID may be used for Notetags.
*
* @help
*
* ============================================================================
* Overview
* ============================================================================
*
* Block Generate is a utility plugin which simulates Block in the style of
* deckbuilders like Slay the Spire. Block is a defensive resource that can be
* accumulated by playing cards or having certain items equipped. When damage
* is dealt to the Actor, Block is deducted first before Health.
*
* This plugin lets you designate states as Block States and Resource States.
*
* Block States allow an Actor or Enemy to build block as a resource and use it
* as part of their Actions. You can configure each Block State to have a 
* different initial and max value, damage type interactions, and log messages.
* Actors can have multiple Block States on them at a given time with each State
* managing its own Block value.
*
* If there are multiple Block States on a target and they take Damage, you can 
* choose the rules by which that Damage is dealt in the Plugin Parameters.
*
* 1) Ascending - Block States with lower IDs will lose value first.
* 2) Descending - Block States with higher IDs will lose value first.
* 3) Even Spread - Damage is equally distributed among all valid Block States.
*
* You can also set rules about who Block States interact with Slip Damage and
* whether a State is removed when its value hits 0 in the plugin parameters.
*
* Resource States are purely for UI where they show a readout of an Actor/Enemy's
* resources such as HP, MP, TP etc. You can also set each Resource States to
* pull its value from a different value associated with the Battler (e.g. ATK, 
* DEF, MAT, MDF, Gold) by changing the value in the Text tab.
*
* Both Block and Resource States are shown in Battle via Icons with Text drawn 
* on them. You can configure the appearance and placement of these Icons in the 
* paramters for each state.
*
* There are a number of Notetags and Script Calls that allow you to configure 
* Block settings for Actors and Enemies. In addition, a number of new Card 
* Actions have been added to let you make Cards that can manipulate the Block 
* values of targets. Please review them below.
*
* ============================================================================
* Card Actions
* ============================================================================
*
* Add Block [State ID] [+/-]X
* Add Block [State Name] [+/-]X
*     Adds X to the value of a Block State for the target of the Card Actions.
*     If an ID is not provided, it defaults to the first Block State in the list.
*     You may use name of the Block State instead of the ID (not case sensitive).
*     X must be a negative or positive whole number.
*     X can be the value stored in a Variable using \v[x] text code.
*
* Set Block [State ID] X
* Set Block [State Name] X
*     Sets the value of a Block State to X for the target of the Card Actions.
*     If an ID is not provided, it defaults to the first Block State in the list.
*     You may use name of the Block State instead of the ID (not case sensitive).
*     X must be a positive whole number.
*     X can be the value stored in a Variable using \v[x] text code.
*
* Fill Block [State ID]
* Fill Block [State Name]
*    Sets the value of a Block State to max for the target of the Card Actions.
*    If an ID is not provided, it defaults to the first Block State in the list.
*    You may use name of the Block State instead of the ID (not case sensitive).
*    Maximum will take the Battler's Maximum for a Block State set by their Notetags.
*    If no maximum was provided, will default to Maximum set in parameters.
*
* Clear Block [State ID]
* Clear Block [State Name]
*     Sets the value of a Block State to 0 for the target of the Card Actions.
*     If an ID is not provided, it defaults to the first Block State in the list.
*     You may use name of the Block State instead of the ID (not case sensitive).
*
* ============================================================================
* Notetags
* ============================================================================
*
* For Actors and Enemies:
*
*   <Init Block [State ID]: X>
*   <Init Block [State Name]: X>
*
* Sets initial value of a Block State of this Actor/Enemy to X at battle start.
* You may use name of the Block State instead of the ID (not case sensitive).
* X must be a negative or positive whole number.
* X can be the value stored in a Variable using \v[x] text code.
*
*   <Max Block [State ID]: X>
*   <Max Block [State Name]: X>
*
* Sets maximum value of a Block State of this Actor/Enemy to X can have at once.
* You may use name of the Block State instead of the ID (not case sensitive).
* X must be a negative or positive whole number.
* X can be the value stored in a Variable using \v[x] text code.
*
*    <Hide Block [State ID]>
*    <Hide Block [State Name]>
*    <Hide All Block>
*
* Hides Icon of a Block State on the Battler's Sprite and Windows.
* If an ID is not provided, it defaults to the first Block State in the list.
* You may use name of the Block State instead of the ID (not case sensitive).
* Hide All Block will hide icons for every ID on the Block State list.
*
*    <Hide Resc [State ID]>
*    <Hide Resc [State Name]>
*    <Hide All Resc>
*
* Hide Icon of a Resource State on the Battler's Sprite and Windows. 
* If an ID is not provided, it defaults to the first Resource State in the list.
* You may use name of the Resource State instead of the ID (not case sensitive).
* Hide All Resc will hide icons for every ID on the Resc State List.
*
* For Skills and Items:
*
*    <Ignore Block [State ID]>
*    <Ignore Block [State Name]>
*
* Damage Formula of these Skills/Items will ignore target's Block State X
* value and deal their damage directly to HP instead. Useful for "piercing" 
* type Skills/Items.
*
* If an ID is not provided, it defaults to the first Block State in the list.
* You may use name of the Block State instead of the ID (not case sensitive).
*
* For Weapons and Armors:
*
*   <Mod Init Block [State ID]: [+/-]X>
*   <Mod Init Block [State Name]: [+/-]X>
*
* Modifies an Actor's Initial for a value Block State by adding Y to it.
* If an ID is not provided, it defaults to the first Block State in the list.
* You may use name of the Block State instead of the ID (not case sensitive).
* X must be a negative or positive whole number.
* X can be the value stored in a Variable using \v[x] text code.
* Can stack from multiple sources (like different pieces of equipment).
*
*   <Mod Max Block [State ID]: [+/-]X>
*   <Mod Max Block [State Name]: [+/-]X>
*
* Modifies an Actor's Max for a Block State value by adding X to it.
* If an ID is not provided, it defaults to the first Block State in the list.
* You may use name of the Block State instead of the ID (not case sensitive).
* X must be a negative or positive whole number.
* X can be the value stored in a Variable using \v[x] text code.
* Can stack from multiple sources (like different pieces of equipment).
*
* For States:
*
*    <Change Block [State ID]: [+/-]X>
*    <Change Block [State Name]: [+/-]X>
*
* Determines how much an Actor with a Block State will gain/lose Block at the 
* end of their turn (during the Regenerate Step).
* If an ID is not provided, it defaults to the first Block State in the list.
* You may use name of the Block State instead of the ID (not case sensitive).
* X must be a negative or positive whole number.
* X can be the value stored in a Variable using \v[x] text code.
* Can stack from multiple sources (like different States on the same Battler).
*
* If there is both Block Gain and Block Loss from multiple States, the values 
* are summed together and applied as a single instance of generation e.g. if 
* State A gains 20 BLOCK each turn and State B loses 40 Block each turn, then 
* an Actor with both State A and B loses 20 Block (+20 - 40 = -20) each turn
*
*    <Fill Block [State ID]>
*    <Fill Block [State Name]>
*
* When a State with this Notetag is applied to a Battler, that Block State value is set to max.
* If an ID is not provided, it defaults to the first Block State in the list.
* You may use name of the Block State instead of the ID (not case sensitive).
* Maximum will take the Battler's Maximum for that Block State set by their Notetags.
* If no maximum was provided, will default to Maximum set in parameters.
*
*    <Clear Block [State ID]>
*    <Clear Block [State Name]>
*
* When a State with this Notetag is applied to a Battler, the Block State value is set to 0.
* If an ID is not provided, it defaults to the first Block State in the list.
* You may use name of the Block State instead of the ID (not case sensitive).
*
*    <Negate Block [State ID]>
*    <Negate Block [State Name]>
*
* When a State with this Notetag is applied to a Battler, Block State is removed and can't be 
* applied to the Battler until this state elapses or is removed. Anything trying to Add or Set
* value to Block State will fail while this State is on the Battler.
*
* If an ID is not provided, it defaults to the first Block State in the list.
* You may use name of the Block State instead of the ID (not case sensitive).
*
* ============================================================================
* Script Calls
* ============================================================================
*
* For Actors and Enemies:
*
*   {battler}.getBlock(id)
*
* Returns the Battler's current value for Block State of specified ID.
* If an ID is not provided, it defaults to the first Block State in the list.
*
*   {battler}.setBlock(id, value, show)
*
* Sets the Battler's value for Block State of specified ID.
* If an ID is not provided, it defaults to the first Block State in the list.
* Value will be processed as a positive or negative whole number.
* Show can be true/false based on if you want a Damage Popup + Log Message for the action.
*
*   {battler}.addBlock(id, value, show)
*
* Adds value to the Battler's Block State of specified ID. 
* If an ID is not provided, it defaults to the first Block State in the list.
* Value will be processed as a positive or negative whole number.
* If it goes over the Battler's Max or under 0, the value is automatically clamped.
* Show can be true/false based on if you want a Damage Popup + Log Message for the action.
*
*   {battler}.fillBlock(id, show)
*
* Fills the Battler's Block State of specified ID setting it to max.
* If an ID is not provided, it defaults to the first Block State in the list.
* Maximum will take the Battler's Maximum for Block State X set by their Notetags.
* If no maximum was provided, will default to Maximum set in parameters.
* Show can be true/false based on if you want a Damage Popup + Log Message for the action.
*
*   {battler}.clearBlock(id, show)
*
* Clears the Battler's Block State of specified ID setting it to 0.
* If an ID is not provided, it defaults to the first Block State in the list.
* Show can be true/false based on if you want a Damage Popup + Log Message for the action.
*
* ============================================================================
* For more information and other features please use the wiki
* that has been made for this and all related plugins:
* http://card-game-combat-help.alwaysdata.net/
* ============================================================================
*
*
* ============================================================================
* Version History
* ============================================================================
*
* v1.3.1 - Improved Message Warning infrastructure.
* 
* v1.3.0  - Changed parameters to accomodate multiple Block/Resource States
*         - Changed Notetags, Card Actions and Script Calls to work with new params
*         - Added Param to turn on/off Battle Log Messages for Block States
*         - Added Param to turn on/off Damge Popups for Block States
*         - Added Param letting you set the Damage Popup colors per Block State
*         - Added Param for whether Slip Damage reduces Block value before HP
*         - Added Param to let you set Resource States to HP, MP, TP or others
*         - Added Params to turn on/off State Icons on Actors and Windows
*         - Block Font Color does not change with Actor HP State in Windows
*         - Allowed using State Names instead of ID for Card Actions and Notetags
*         - Changed <Block Change> Notetag syntax to <Change Block>
*         - Added <Fill Block> and <Clear Block> State Notetags
*         - Added support for Variable Input in Card Actions and Notetags
*
* v1.2.0  - Fixed issue with Normal Attack damage type ignoring Block
*         - <Ignore Block> is working again and unaffected by Damage Types
*         - Added new Card Actions for Add Block, Set Block and Clear Block
*         - Fixed bug where Block carried over rounds despite showing it cleared
*
*
* v1.1.0 - Added in Health Icon in case users want to show both Health and 
*          Block on a Battler. Has a Notetags if you want to hide it.
*        - Block and Health values are written to Icons on Actor & Party Windows
*          Can set different Fonts for Block and Health Icon text.
*        - Exposed the text for State Messages for Block and Break. Now can 
*          be customized. Has differences between MV and MZ.
*        - Added support for Elements. You can now choose which Damage Types
*          the Block State reduces vs which ones are ignored.
*        - Different Icon Offsets can be declared for Actors and Enemies
*
* v1.0.0 - Base Functionality complete! Will probably come back around and do
*          some minor edits once I am done testing this with CGC Demo Project.
*
*
* ============================================================================
* Contact Info
* ============================================================================
*
* This tool was developed by folks at MythAtelier LLC. We make Games that Care.
*
* Need more tools for your project? Be sure to check out our other plugins here:
* https://itch.io/c/1695699/tools-plugins
*
* Have any questions? Run into any bugs? Want to chat? Best place to reach us:
* https://discord.gg/wRk4XHF5tZ
*
* If you like this plugin and want to support us, please give our Patreon a look:
* https://www.patreon.com/mythatelier
*
* Other Places to Find Us:
* - YouTube: https://www.youtube.com/channel/UCQn_1kWmj2iAkNEv9eqL_LA
* - BSky: https://bsky.app/profile/mythatelier.itch.io
* - Steam: https://store.steampowered.com/search/?developer=MythAtelier%2C%20LLC
*
*/

/*~struct~Coordinate:
 * @param x
 * @text X Coordinate
 * @type number
 * @min -10000
 * @default 0
 *
 * @param y
 * @text Y Coordinate
 * @type number
 * @min -10000
 * @default 0
 *
 */

/*~struct~Icon:
* @param showIconOnBattler
* @text Show on Battlers?
* @type select
* @option Hide for all Battlers
* @option Show only for Actors
* @option Show only for Enemies
* @option Show for all Battlers
* @default Show for all Battlers
*
* @param iconOffActor
* @text Actor Offset
* @type struct<Coordinate>
* @default {"x":"0","y":"0"}
* @desc X/Y Offsets for the Icon relative to its base position at the Actor.
*
* @param iconOffEnemy
* @text Enemy Offset
* @type struct<Coordinate>
* @default {"x":"0","y":"0"}
* @desc X/Y Offsets for the Icon relative to its base position at the Enemy.
*
* @param showIconOnWindow
* @text Show in Windows?
* @type select
* @option Hide in Both Status Windows
* @option Show only in Actor Status Window
* @option Show only in Party Status Window
* @option Show in Both Status Windows
* @desc Settings for showing/hiding the Icon in Actor and Party Status Windows.
* @default Show in Both Status Windows
*
* @param iconFontFace
* @parent Block State Settings
* @text Font Face
* @desc Font Face for the value drawn on the Icon
* @type string
* @default GameFont
*
* @param iconFontSize
* @parent Block State Settings
* @text Font Size
* @desc Font Size for the value drawn on the Icon
* @type number
* @default 18 
*
*/

/*~struct~BlockState:
* @param blockStateID
* @text State ID
* @type state
* @desc State ID of the Block State. If set to None/0, this state will not be valid.
* @default 1
*
* @param defaultInitBlock
* @type number
* @text Default Initial Value
* @desc Default Initial Block value of this Block State set for all Battlers at Battle Start. If set higher than Max, will be clamped.
* @default 0
* @min 0
*
* @param defaultMaxBlock
* @type number
* @text Default Maximum Value
* @desc Default Maximum Block value of this Block State set for all Battlers at Battle Start. Set to -1 to have no Maximum.
* @default -1
*
* @param blockIcon
* @text Icon Settings
* @type struct<Icon>
* @default {"showIconOnBattler":"Show for all Battlers","iconOffActor":"{\"x\":\"0\",\"y\":\"0\"}","iconOffEnemy":"{\"x\":\"0\",\"y\":\"0\"}","showIconOnWindow":"Show in Both Status Windows","iconFontFace":"GameFont","iconFontSize":"18"}
*
* @param blockMsgs
* @text Log Messages
* @type struct<BlockMsgs>
* @default {"showBlockMessages":"true","gainText":"\"%1 loses %2 %3\"","loseText":"\"%1 loses %2 %3\"","blockText":"\"BLOCK! %1 blocked %2 damage!\"","breakText":"\"BREAK! %1 took %2 damage!\""}
*
* @param blockedTypes
* @text Blocked Elements
* @desc List of Elements blocked by State e.g "1, 2, 3" blocks Damage of Elements 1, 2, 3 while the rest will bypass.
* @type string
* @default "1"
*
* @param allowDamagePopup
* @text Allow Damage Popup?
* @type boolean
* @desc If ON, a Damage Popup is generated to reflect change in Block State. If OFF, no Popup is generated.
* @default true
*
* @param popupColor
* @text Popup Color Code
* @type number
* @min 0
* @desc Set the Color of your Damage Popup (for MV, matches with your Damage numbers spritesheet in img/system)
* @default 2
*/

/*~struct~BlockMsgs:
* @param showBlockMessages
* @text Show Messages in Log?
* @type boolean
* @desc If ON, all Messages below will show in the Battle Log. If OFF, none will show.
* @default true
*
* @param gainText
* @text Value Gain Text
* @type string
* @default "%1 gains %2 %3."
* @desc Log Text Popup when you gain Block Value via Card Action.   %1 - Subject, %2 - Value, %3 - State Name
*
* @param loseText
* @text Value Loss Text
* @type string
* @default "%1 loses %2 %3."
* @desc Log Text Popup when you lose Block Value via Card Action.   %1 - Subject, %2 - Value, %3 - State Name
*
* @param blockText
* @text Block State Message
* @type string
* @default "BLOCK! %1 blocked %2 damage!"
* @desc Log Text Popup when the Block State Holds after Damage.   %1 - Target, %2 - Damage, %3 - State Name
*
* @param breakText
* @text Break State Message
* @type string
* @default "BREAK! %1 took %2 damage!"
* @desc Log Text Popup when the Block State Breaks after Damage.   %1- Target, %2 - Damage, %3 - State Name
*
*/

/*~struct~RescState:
* @param rescStateID
* @text State ID
* @type state
* @desc State of the Resource State. If set to None/0, this state will not be valid.
* @default 1
*
* @param rescType
* @text Resource Type
* @type select
* @option HP
* @option MP
* @option TP
* @desc Choose if Icon shows the Battler's HP, MP or TP. Use Text tab for parameters such as ATK, DEF, MAT etc.
* @default HP
*
* @param rescIcon
* @text Icon Settings
* @type struct<Icon>
* @default {"showIconOnBattler":"Show for all Battlers","iconOffActor":"{\"x\":\"0\",\"y\":\"0\"}","iconOffEnemy":"{\"x\":\"0\",\"y\":\"0\"}","showIconOnWindow":"Show in Both Status Windows","iconFontFace":"GameFont","iconFontSize":"18"}
*
*/

var Myth = Myth || {};
Myth.Util = Myth.Util || {};
Myth.Util.usingMZ = (Utils.RPGMAKER_NAME == "MZ");
Myth.Util.spritePrototype = Myth.Util.usingMZ ? Sprite_Clickable : Sprite_Base;

Myth.BLK = Myth.BLK || {};

//=============================================================================
// Warn Messages
//=============================================================================
// #region Warn Messages

Myth.BLK.warnMessage = function (index, ...args)
{
    if ($dataSystem == undefined)
    {
        return setTimeout(() => {
            this.warnMessage(index, ...args);
        }, 100);
    }
    const languagePrefix = $dataSystem.locale.slice(0, 2).toLowerCase();
    let text = "";
    let warnMessageMap = {
        ja: this.warnMessageJa,		// Japanese
        zh: this.warnMessageZh,		// Chinese
        ko: this.warnMessageKo,		// Korean
        en: this.warnMessageEn		// English
    };
    if (warnMessageMap[languagePrefix])
        text = warnMessageMap[languagePrefix].call(this, index, ...args);
    else
    {
        text = this.warnMessageEn(index, ...args);	// Default to English for any unsupported language.
    }

    console.warn(text);
    console.warn("MythAtelier Error Code: BLK " + index);  // Error code should be the same in every language.
    Myth.Util.openConsole();
    return text;
}

Myth.BLK.warnMessageEn = function (index, ...args)
{
    let text = "";
    switch (index)
    {
        case 0: // ????
        {
            text = "No Block States defined in the Block Generate plugin parameters. Please add at least one Block State for Add Block to work.";
            break;
        }
        case 1: // ????
        {
            text = "No Block States defined in the Block Generate plugin parameters. Please make changes to that Card Action.";
            break;
        }
        case 2: // ????
        {
            text = "No Block State with that Name was found. Is the Name given the same as in the Database? (could be case sensitive)";
            break;
        }
        case 3: // ????
        {
            text = "No Block States defined in the Block Generate plugin parameters. Please add at least one Block State for Set Block to work.";
            break;
        }
    }

    return text;
}

Myth.BLK.warnMessageJa = function (index, ...args)
{
	return this.warnMessageEn(index, ...args); // Japanese translation not available yet.
}

Myth.BLK.warnMessageZh = function (index, ...args)
{
	return this.warnMessageEn(index, ...args); // Chinese translation not available yet.
}

Myth.BLK.warnMessageKo = function (index, ...args)
{
	return this.warnMessageEn(index, ...args); // Korean translation not available yet.
}

// #endregion

Myth.Parameters = PluginManager.parameters('Myth_CGC_BlockGenerate');

Myth.BLK._blockStatesList = []; var bsList = [];

try { bsList = JSON.parse(PluginManager.parameters('Myth_CGC_BlockGenerate').blockStatesList); }
catch (error) {
    console.warn("No valid Block States defined in MYTH_CGC_BlockGenerate plugin parameters. Please define at least one or you won't be able to use its functions.");
};

for (var i = 0; i < bsList.length; i++)
{
    var obj = JSON.parse(bsList[i]);
    for (const property in obj) obj[property] = JSON.parse(obj[property]);
    //sucks to have to do this manually, need a better method for parsing
    obj.blockedTypes = String(obj.blockedTypes).split('"').join('').split(",").map(Number);
    obj.blockMsgs.blockText = String(obj.blockMsgs.blockText).split('"').join('');
    obj.blockMsgs.breakText = String(obj.blockMsgs.breakText).split('"').join('');
    obj.blockedTypes.push(-1);    //exception for Normal Attack DMG element, which for whatever reason is -1
    obj.blockIcon.iconFontSize = Number(obj.blockIcon.iconFontSize);
    obj.blockIcon.iconOffActor = JSON.parse(obj.blockIcon.iconOffActor);
    obj.blockIcon.iconOffActor.x = Number(obj.blockIcon.iconOffActor.x);
    obj.blockIcon.iconOffActor.y = Number(obj.blockIcon.iconOffActor.y);
    obj.blockIcon.iconOffEnemy = JSON.parse(obj.blockIcon.iconOffEnemy);
    obj.blockIcon.iconOffEnemy.x = Number(obj.blockIcon.iconOffEnemy.x);
    obj.blockIcon.iconOffEnemy.y = Number(obj.blockIcon.iconOffEnemy.y);
    Myth.BLK._blockStatesList.push(obj);
};

Myth.BLK.multiBlockRule = PluginManager.parameters('Myth_CGC_BlockGenerate').multiBlockRule;
Myth.BLK.allowSlipDMG = JSON.parse(Myth.Parameters.allowSlipDMG);
Myth.BLK.removeStateOnZero = JSON.parse(Myth.Parameters.removeStateOnZero);

Myth.BLK._rescStatesList = [];  var rsList = [];

try { rsList = JSON.parse(PluginManager.parameters('Myth_CGC_BlockGenerate').rescStatesList); }
catch (error) {
    console.warn("No valid Resource States defined in MYTH_CGC_BlockGenerate plugin parameters. Please define at least one or you won't be able to use its functions.");
}

for (var i = 0; i < rsList.length; i++)
{
    var obj = JSON.parse(rsList[i]);
    obj.rescIcon = JSON.parse(obj.rescIcon);
    //sucks to have to do this manually, need a better method for parsing
    obj.rescIcon.iconFontSize = Number(obj.rescIcon.iconFontSize);
    obj.rescIcon.iconOffActor = JSON.parse(obj.rescIcon.iconOffActor);
    obj.rescIcon.iconOffActor.x = Number(obj.rescIcon.iconOffActor.x);
    obj.rescIcon.iconOffActor.y = Number(obj.rescIcon.iconOffActor.y);
    obj.rescIcon.iconOffEnemy = JSON.parse(obj.rescIcon.iconOffEnemy);
    obj.rescIcon.iconOffEnemy.x = Number(obj.rescIcon.iconOffEnemy.x);
    obj.rescIcon.iconOffEnemy.y = Number(obj.rescIcon.iconOffEnemy.y);
    obj.rescStateID = Number(obj.rescStateID);
    Myth.BLK._rescStatesList.push(obj);
};

//=============================================================================
// Utility Functions
//=============================================================================

Myth.Util.clamp = function(value, min, max)
{
    return Math.min(Math.max(value, min), max);
}

Myth.Util.getStateName = function (id)
{
    return $dataStates[id].name;
}

Myth.Util.getStateID = function (name)
{
    return $dataStates.map(function (e) { if (e) { return e.name; } }).indexOf(name);
}

Myth.BLK.windowRefresh = function ()
{
    if (SceneManager._scene._statusWindow) SceneManager._scene._statusWindow.refresh();
    if (SceneManager._scene._partyStatusWindow) SceneManager._scene._partyStatusWindow.refresh();
    if (SceneManager._scene._actorWindow) SceneManager._scene._actorWindow.refresh();
}

Myth.BLK.isBlockStateValid = function (stateId)
{
    var isValid = false;
    Myth.BLK._blockStatesList.map(a => { if (a.blockStateID == stateId) isValid = true; });
    return isValid;
}

Myth.BLK.isRescStateValid = function (stateId)
{
    var isValid = false;
    Myth.BLK._rescStatesList.map(a => { if (a.rescStateID == stateId) isValid = true; });
    return isValid;
}

Myth.BLK.isStateNegated = function (battler, stateId)
{
    var isNegated = false;

    battler.states().forEach(function (state)
    {
        for (var i = 0; i < Myth.BLK._blockStatesList.length; i++)
        {
            if ($dataStates[state.id]._negateBlock[Myth.BLK._blockStatesList[i].blockStateID]) isNegated = true;
        }
    }, this);

    return isNegated;
}

Myth.BLK.getBlockStateInfo = function (stateId)
{
    var objIndex = Myth.BLK._blockStatesList.map(function (e) { if (e) { return e.blockStateID; } }).indexOf(stateId);
    if (objIndex == -1) objIndex = 0;
    return Myth.BLK._blockStatesList[objIndex];
}

Myth.BLK.getRescStateInfo = function (stateId)
{
    var objIndex = Myth.BLK._rescStatesList.map(function (e) { if (e) { return e.rescStateID; } }).indexOf(stateId);
    if (objIndex == -1) objIndex = 0;
    return Myth.BLK._rescStatesList[objIndex];
}

//=============================================================================
// Game_Battler
//=============================================================================

Myth.BLK.GameBattler_initialize = Game_Battler.prototype.initialize;
Game_Battler.prototype.initialize = function ()
{
    Myth.BLK.GameBattler_initialize.call(this);

    this._blockCount = [];
    this._initBlock = [];
    this._maxBlock = [];
    this._lastBlocked = [];
}

Myth.BLK._GameBattler_rengerateAll = Game_Battler.prototype.regenerateAll;
Game_Battler.prototype.regenerateAll = function() {
	Myth.BLK._GameBattler_rengerateAll.call(this);

	if(this.isAlive())
	{
        for (var i = 0; i < Myth.BLK._blockStatesList.length; i++)
        {
            const bState = Myth.BLK._blockStatesList[i];

            var blockChangeList = [];

            this.states().forEach(function (state)
            {
                if ($dataStates[state.id]._blockChanges[bState.blockStateID])
                {
                    blockChangeList.push($dataStates[state.id]._blockChanges[bState.blockStateID]);
                }
            }, this);

            this.addBlock(bState.blockStateID, blockChangeList.reduce((total, curr) => total + curr, 0), false);
        }
	}
};

Myth.BLK._GameBattler_rengerateHp = Game_Battler.prototype.regenerateHp;

Game_Battler.prototype.regenerateHp = function ()
{
    if (!Myth.BLK.allowSlipDMG)
    {
        Myth.BLK._GameBattler_rengerateHp.call(this);
    }
    else
    {
        var value = Math.floor(this.mhp * this.hrg);
        value = Math.max(value, -this.maxSlipDamage());

        //Block doesn't affect HP Regen, only Slip Damage/HP Degen
        if (value < 0)
        {
            var skipBlock = false;

            var blockStates = this._states.filter(a => Myth.BLK.isBlockStateValid(a));
            if (blockStates.length == 0) skipBlock = true;

            if (!skipBlock)
            {
                switch (Myth.BLK.multiBlockRule)
                {
                    case "Ascending": blockStates = blockStates.sort(function (a, b) { return a - b }); break;
                    case "Descending": blockStates = blockStates.sort(function (a, b) { return b - a }); break;
                    case "Even Spread": blockStates = blockStates.filter(a => this._blockCount[a] > 0); break;
                }

                if (Myth.BLK.multiBlockRule != "Even Spread")
                {
                    for (var i = 0; i < blockStates.length; i++)
                    {
                        if (this.getBlock(blockStates[i]) > Math.abs(value))
                        {
                            if (Myth.BLK.getBlockStateInfo(blockStates[i]).allowDamagePopup) this.startBlockPopup(blockStates[i], value);
                            this._blockCount[blockStates[i]] -= Math.abs(value);
                            this._lastBlocked = { id: blockStates[i], amt: Math.abs(value) };
                            value = 0;
                        }
                        else if (this.getBlock(blockStates[i]) <= Math.abs(value))
                        {
                            if (Myth.BLK.getBlockStateInfo(blockStates[i]).allowDamagePopup) this.startBlockPopup(blockStates[i], this._blockCount[blockStates[i]]);
                            value += this._blockCount[blockStates[i]];
                            this._lastBlocked = { id: blockStates[i], amt: this._blockCount[blockStates[i]] };
                            this._blockCount[blockStates[i]] = 0;
                            this.clearBlock(blockStates[i], true);
                        }

                        if (value == 0) break;
                    }
                }
                else
                {
                    var totalBlock = 0;
                    for (var i = 0; i < blockStates.length; i++) totalBlock += this.getBlock(blockStates[i]);

                    if (totalBlock > Math.abs(value))
                    {
                        var valueAmt = Math.floor(Math.abs(value) / blockStates.length);

                        for (var i = 0; i < blockStates.length; i++)
                        {
                            if (Myth.BLK.getBlockStateInfo(blockStates[i]).allowDamagePopup) this.startBlockPopup(blockStates[i], valueAmt);
                            this._blockCount[blockStates[i]] -= valueAmt;
                            this._lastBlocked = { id: blockStates[i], amt: Math.abs(valueAmt) };
                        }

                        value = 0;
                    }
                    else if (totalBlock <= Math.abs(value))
                    {
                        for (var i = 0; i < blockStates.length; i++)
                        {
                            if (Myth.BLK.getBlockStateInfo(blockStates[i]).allowDamagePopup) this.startBlockPopup(blockStates[i], this._blockCount[blockStates[i]]);
                            value += this._blockCount[blockStates[i]];
                            this._lastBlocked = { id: blockStates[i], amt: this._blockCount[blockStates[i]] };
                            this.clearBlock(blockStates[i], true);
                        }
                    }
                }
            }
        }

        Myth.BLK.windowRefresh();
        if (value !== 0) this.gainHp(value);
    }
};

Game_Battler.prototype.equipsInitMods = function(stateId)
{
    if (this.isActor() && this.isBlockStateValid(stateId))
    {
        for(var i = 0; i < this._equips.length; ++i)
        {
            if (this._equips[i]._itemId != 0)
            {
                var mod = (this._equips[i]._dataClass == "weapon") ?
                    $dataWeapons[this._equips[i]._itemId]._initMods[stateId] :
                    $dataArmors[this._equips[i]._itemId]._initMods[stateId];

                if (mod) this._initBlock[stateId] += mod;
            }
        }
    }
}

Game_Battler.prototype.equipsMaxMods = function(stateId)
{
    if (this.isActor() && this.isBlockStateValid(stateId))
    {
        for(var i = 0; i < this._equips.length; ++i)
        {
            if (this._equips[i]._itemId != 0)
            {
                var mod = (this._equips[i]._dataClass == "weapon") ?
                    $dataWeapons[this._equips[i]._itemId]._maxMods[stateId] :
                    $dataArmors[this._equips[i]._itemId]._maxMods[stateId];

                if (mod) this._maxBlock[stateId] += mod;
            }
        }
    }
}

Game_Battler.prototype.getBlock = function(stateId)
{
    if (Myth.BLK._blockStatesList.length > 0)
    {
        if (this.isBlockStateValid(stateId))
        {
            return this._blockCount[stateId];
        }
        else
        {
            return this._blockCount[Myth.BLK._blockStatesList[0].blockStateID];
        }
    }

   return 0;
}

Game_Battler.prototype.setBlock = function (stateId, value, show)
{
    if (!this.isBlockStateValid(stateId)) return;
    if (Myth.BLK.isStateNegated(this, stateId)) return;

    if (!this.isStateAffected(stateId))
    {
        this._lastBlocked = this._lastBlocked = { id: stateId, amt: value };
        this.addState(stateId);
    }

    this._blockCount[stateId] = Myth.Util.clamp(value, 0, this._maxBlock[stateId]);

    if (show && Myth.BLK.getBlockStateInfo(stateId).allowDamagePopup) this.startBlockPopup(stateId, value);

    Myth.BLK.windowRefresh();
}

Game_Battler.prototype.isBlockStateValid = function (stateId)
{
    return Myth.BLK.isBlockStateValid(stateId);
}

Game_Battler.prototype.addBlock = function (stateId, value, show)
{
    if (!this.isBlockStateValid(stateId)) return;
    if (value == 0) return;
    if (Myth.BLK.isStateNegated(this, stateId)) return;

    this._blockCount[stateId] = Myth.Util.clamp(this._blockCount[stateId] + value, 0, this._maxBlock[stateId]);
    if (!this.isStateAffected(stateId) && this._blockCount[stateId] > 0) this.addState(stateId);
    if (this._blockCount[stateId] <= 0) this.clearState(stateId);

    this._lastBlocked = { id: stateId, amt: value };

    if (show)
    {
        if (Myth.BLK.getBlockStateInfo(stateId).allowDamagePopup) this.startBlockPopup(stateId, value);
        this.startBlockLogMsg(stateId, value);
    }

    Myth.BLK.windowRefresh();
}

Game_Battler.prototype.fillBlock = function (stateId, show)
{
    if (!this.isBlockStateValid(stateId)) return;

    if (show && Myth.BLK.getBlockStateInfo(stateId).allowDamagePopup) this.startBlockPopup(stateId, this._maxBlock[stateId] - this._blockCount[stateId]);

    this._blockCount[stateId] = this._maxBlock[stateId];
    this._lastBlocked = { id: stateId, amt: this._maxBlock[stateId] };
    if (!this.isStateAffected(stateId)) this.addState(stateId);

    Myth.BLK.windowRefresh();    
}

Game_Battler.prototype.clearBlock = function (stateId, show)
{
    if (!this.isBlockStateValid(stateId)) return;

    if (show && Myth.BLK.getBlockStateInfo(stateId).allowDamagePopup) this.startBlockPopup(stateId, this._blockCount[stateId]);

    this._blockCount[stateId] = 0;
    this._lastBlocked = { id: stateId, amt: 0 };
    if (this.isStateAffected(stateId) && Myth.BLK.removeStateOnZero) this.removeState(stateId);

    Myth.BLK.windowRefresh();
}

Game_Battler.prototype.startBlockPopup = function(stateId, value)
{
    if (this.isActor() ? $dataActors[this._actorId]._hideBlock[stateId] : $dataEnemies[this._enemyId]._hideBlock[stateId]) return;

    var blockPop = new Sprite_Damage();

    var colorCode = Myth.BLK.getBlockStateInfo(stateId).popupColor || 2;

    if(Myth.Util.usingMZ)
    {
        blockPop._colorType = colorCode;
        blockPop.createDigits(value);
    }
    else
    {
        blockPop.createDigits(colorCode, value);
    }

    var spriteArray = (this.isActor()) ? BattleManager._spriteset._actorSprites : BattleManager._spriteset._enemySprites;
    var battlerSprite = spriteArray.filter(obj => {return obj._battler === this}).shift();

    if(battlerSprite && this.isSpriteVisible())
    {
        battlerSprite._damages.push(blockPop);
        blockPop.x = battlerSprite.x + battlerSprite.damageOffsetX();
        blockPop.y = battlerSprite.y + battlerSprite.damageOffsetY() - (battlerSprite._damages[0].isPlaying() ? 48 : 0);
        battlerSprite.parent.addChild(blockPop);
    }
}

Game_Battler.prototype.startBlockLogMsg = function(stateId, value)
{
    var battleLog = SceneManager._scene._logWindow;

    if (battleLog && BattleManager._subject)
    {
        battleLog.push('popBaseLine');
        battleLog.push('pushBaseLine');

        var text = "";

        if (value >= 0)
        {
            text = String(Myth.BLK.getBlockStateInfo(stateId).blockMsgs.gainText);
        }
        else
        {
            text = String(Myth.BLK.getBlockStateInfo(stateId).blockMsgs.loseText);
        }

        text = text.replace(/%1/g, BattleManager._subject.name());
        text = text.replace(/%2/g, String(value));
        text = text.replace(/%3/g, String($dataStates[stateId].name));
        text = text.replace(/['"]+/g, '');

        battleLog.push('addText', text);            
        battleLog.push('waitForEffect');
    }
}

Myth.BLK._GameBattler_addState = Game_Battler.prototype.addState;
Game_Battler.prototype.addState = function(stateId) {
    if(this.isStateAddable(stateId))
    {
        for (var i = 0; i < Myth.BLK._blockStatesList.length; i++)
        {
            const bState = Myth.BLK._blockStatesList[i];
            if ($dataStates[stateId]._fillBlock[bState.blockStateID]) this.fillBlock(bState.blockStateID, true);
            if ($dataStates[stateId]._clearBlock[bState.blockStateID]) this.clearBlock(bState.blockStateID, true);
            if ($dataStates[stateId]._negateBlock[bState.blockStateID]) this.clearBlock(bState.blockStateID, true);
        }
    }
    Myth.BLK._GameBattler_addState.call(this, stateId);
};

Myth.BLK._GameBattler_removeState = Game_Battler.prototype.removeState;

Game_Battler.prototype.removeState = function (stateId)
{
    Myth.BLK._GameBattler_removeState.call(this, stateId);
    if (Myth.BLK.isBlockStateValid(stateId)) this.clearBlock(stateId);
}

//=============================================================================
// BattleManager
//=============================================================================

Myth.BLK._BattleManager_startBattle = BattleManager.startBattle;
BattleManager.startBattle = function() {
    Myth.BLK._BattleManager_startBattle.call(this);

    $gameParty.battleMembers().forEach((actor) => 
    { 
        for (var i = 0; i < Myth.BLK._blockStatesList.length; i++)
        {
            const bsList = Myth.BLK._blockStatesList[i];
            if (bsList)
            {
                actor._blockCount[bsList.blockStateID] = 0;
                actor._initBlock[bsList.blockStateID] = bsList.defaultInitBlock || 0;

                if ($dataActors[actor._actorId]._initBlock[bsList.blockStateID])
                    actor._initBlock[bsList.blockStateID] = $dataActors[actor._actorId]._initBlock[bsList.blockStateID];

                actor.equipsInitMods(bsList.blockStateID);

                actor._maxBlock[bsList.blockStateID] = bsList.defaultMaxBlock || 100;
                if ($dataActors[actor._actorId]._maxBlock[bsList.blockStateID])
                    actor._maxBlock[bsList.blockStateID] = $dataActors[actor._actorId]._maxBlock[bsList.blockStateID];

                actor.equipsMaxMods(bsList.blockStateID);

                if (actor._maxBlock[bsList.blockStateID] == -1) actor._maxBlock[bsList.blockStateID] = 999999;

                actor._lastBlocked = { id: bsList.blockStateID, amt: 0 };
                actor.setBlock(bsList.blockStateID, actor._initBlock[bsList.blockStateID],false);
            }
        }

        for (var k = 0; k < Myth.BLK._rescStatesList.length; k++)
        {
            const rsList = Myth.BLK._rescStatesList[k];
            if (rsList) actor.addState(rsList.rescStateID);
        }
    });

    $gameTroop.members().forEach((enemy) =>
    {
        for (var j = 0; j < Myth.BLK._blockStatesList.length; j++)
        {
            const bsList = Myth.BLK._blockStatesList[j];
            if (bsList)
            {
                enemy._blockCount[bsList.blockStateID] = 0;
                enemy._initBlock[bsList.blockStateID] = bsList.defaultInitBlock || 0;

                if ($dataEnemies[enemy._enemyId]._initBlock[bsList.blockStateID])
                    enemy._initBlock[bsList.blockStateID] = $dataEnemies[enemy._enemyId]._initBlock[bsList.blockStateID];

                enemy._maxBlock[bsList.blockStateID] = bsList.defaultMaxBlock || 100;
                if ($dataEnemies[enemy._enemyId]._maxBlock[bsList.blockStateID])
                    enemy._maxBlock[bsList.blockStateID] = $dataEnemies[enemy._enemyId]._maxBlock[bsList.blockStateID];

                if (enemy._maxBlock[bsList.blockStateID] == -1) enemy._maxBlock[bsList.blockStateID] = 999999;

                enemy._lastBlocked = { id: bsList.blockStateID, amt: 0 };
                enemy.setBlock(bsList.blockStateID, enemy._initBlock[bsList.blockStateID], false);
            }
        }

        for (var k = 0; k < Myth.BLK._rescStatesList.length; k++)
        {
            const rsList = Myth.BLK._rescStatesList[k];
            if (rsList) enemy.addState(rsList.rescStateID);
        }
    });

};

//=============================================================================
// Sprite_HealthIcon
//=============================================================================

function Sprite_HealthIcon(){
    this.initialize.apply(this, arguments);
}

Sprite_HealthIcon.prototype = Object.create(Sprite_StateIcon.prototype);
Sprite_HealthIcon.prototype.constructor = Sprite_HealthIcon;

Sprite_HealthIcon.prototype.initialize = function (stateId)
{
    Sprite.prototype.initialize.call(this);
    this.initMembers(stateId);
    this.loadBitmap();
}

Sprite_HealthIcon.prototype.initMembers = function(stateId)
{
    Sprite_StateIcon.prototype.initMembers.call(this);

    this._blockReadout = new Sprite();
    this.addChild(this._blockReadout);

    this._origX = this.x;
    this._origY = this.y;
    this._stateId = stateId;

    this._blockReadout.anchor.x = 0.5;
    this._blockReadout.anchor.y = 0.5;

    this._blockReadout.bitmap = (Utils.RPGMAKER_NAME == "MZ") ?
        new Bitmap(ImageManager.iconWidth, ImageManager.iconHeight) :
        new Bitmap(Window_Base._iconWidth, Window_Base._iconHeight);

    this._blockReadout.bitmap.fontSize = Myth.BLK.getRescStateInfo(stateId).rescIcon.iconFontSize;
    this._blockReadout.bitmap._fontFace = Myth.BLK.getRescStateInfo(stateId).rescIcon.iconFontFace;
}

Sprite_HealthIcon.prototype.update = function()
{
    Sprite.prototype.update.call(this);

    this.updateIcon();
    this.updateFrame();
}

Sprite_HealthIcon.prototype.updateIcon = function()
{
    var contents = this._blockReadout.bitmap;
    contents.clear();

    if(this._battler && this._battler.isAlive() && this._battler.isStateAffected(this._stateId))
    {
        if ((this._battler.isActor() ? $dataActors[this._battler._actorId] :
            $dataEnemies[this._battler._enemyId])._hideResc[this._stateId]) return;

        var objIndex = Myth.BLK._rescStatesList.map(function (e) { if (e) { return e.rescStateID; } }).indexOf(this._stateId);
        if (objIndex == -1) objIndex = 0;

        var objIcon = Myth.BLK._rescStatesList[objIndex].rescIcon;

        this.x = this._origX + (this._battler.isActor() ? objIcon.iconOffActor.x : objIcon.iconOffEnemy.x);
        this.y = this._origY + (this._battler.isActor() ? objIcon.iconOffActor.y : objIcon.iconOffEnemy.y);

        this._iconIndex = $dataStates[this._stateId].iconIndex;

        var evalValue = eval("this._battler." + String(Myth.BLK._rescStatesList[objIndex].rescType).toLowerCase());

        contents.drawText(String(evalValue), 0, 0,
        (Utils.RPGMAKER_NAME == "MZ") ? ImageManager.iconWidth : Window_Base._iconWidth, 
        (Utils.RPGMAKER_NAME == "MZ") ? ImageManager.iconHeight : Window_Base._iconHeight, 'center');
    }
    else
    {
        this._iconIndex = 0;
    }
}

//=============================================================================
// Sprite_BlockIcon
//=============================================================================

function Sprite_BlockIcon(){
    this.initialize.apply(this, arguments);
}

Sprite_BlockIcon.prototype = Object.create(Sprite_StateIcon.prototype);
Sprite_BlockIcon.prototype.constructor = Sprite_BlockIcon;

Sprite_BlockIcon.prototype.initialize = function (stateId)
{
    Sprite.prototype.initialize.call(this);
    this.initMembers(stateId);
    this.loadBitmap();
}

Sprite_BlockIcon.prototype.initMembers = function(stateId)
{
    Sprite_StateIcon.prototype.initMembers.call(this);

    this._blockReadout = new Sprite();
    this.addChild(this._blockReadout);

    this._origX = this.x;
    this._origY = this.y;
    this._stateId = stateId;

    this._blockReadout.anchor.x = 0.5;
    this._blockReadout.anchor.y = 0.5;

    this._blockReadout.bitmap = (Utils.RPGMAKER_NAME == "MZ") ?
        new Bitmap(ImageManager.iconWidth, ImageManager.iconHeight) :
        new Bitmap(Window_Base._iconWidth, Window_Base._iconHeight);

    this._blockReadout.bitmap.fontSize = Myth.BLK.getBlockStateInfo(stateId).blockIcon.iconFontSize;
    this._blockReadout.bitmap._fontFace = Myth.BLK.getBlockStateInfo(stateId).blockIcon.iconFontFace;
}

Sprite_BlockIcon.prototype.update = function()
{
    Sprite.prototype.update.call(this);

    this.updateIcon();
    this.updateFrame();
}

Sprite_BlockIcon.prototype.updateIcon = function()
{
    var contents = this._blockReadout.bitmap;
    contents.clear();

    if (this._battler && this._battler.isAlive() && this._battler.isStateAffected(this._stateId) && this._battler.isBlockStateValid(this._stateId))
    {
        if ((this._battler.isActor() ? $dataActors[this._battler._actorId] :
            $dataEnemies[this._battler._enemyId])._hideBlock[this._stateId]) return;

        var objIndex = Myth.BLK._blockStatesList.map(function (e) { if (e) { return e.blockStateID; } }).indexOf(this._stateId);
        if (objIndex == -1) objIndex = 0;

        var objIcon = Myth.BLK._blockStatesList[objIndex].blockIcon;

        this.x = this._origX + (this._battler.isActor() ? objIcon.iconOffActor.x : objIcon.iconOffEnemy.x);
        this.y = this._origY + (this._battler.isActor() ? objIcon.iconOffActor.y : objIcon.iconOffEnemy.y);

        this._iconIndex = $dataStates[this._stateId].iconIndex;
        contents.drawText(String(this._battler.getBlock(this._stateId)), 0, 0,
        (Utils.RPGMAKER_NAME == "MZ") ? ImageManager.iconWidth : Window_Base._iconWidth, 
        (Utils.RPGMAKER_NAME == "MZ") ? ImageManager.iconHeight : Window_Base._iconHeight, 'center');
    }
    else
    {
        this._iconIndex = 0;
    }
}

Myth.BLK._BlockStateIcon_updateIcon = Sprite_StateIcon.prototype.updateIcon;
Sprite_StateIcon.prototype.updateIcon = function()
{
    Myth.BLK._BlockStateIcon_updateIcon.call(this);
    //this may cause issues if two states reuse the same icon

    for (var i = 0; i < Myth.BLK._blockStatesList.length; i++)
    {
        const bsList = Myth.BLK._blockStatesList[i];
        if (bsList)
        {
            if (this._iconIndex == $dataStates[bsList.blockStateID].iconIndex) this._iconIndex = 0;
        }
    }

    for (var i = 0; i < Myth.BLK._rescStatesList.length; i++)
    {
        const rsList = Myth.BLK._rescStatesList[i];
        if (rsList)
        {
            if (this._iconIndex == $dataStates[rsList.rescStateID].iconIndex) this._iconIndex = 0;
        }
    }
}

//=============================================================================
// Sprite_Battler
//=============================================================================

Myth.BLK._SpriteActor_initMembers = Sprite_Actor.prototype.initMembers;
Sprite_Actor.prototype.initMembers = function() {
    Myth.BLK._SpriteActor_initMembers.call(this);
    if($dataSystem.optSideView) this.createBlockIconSprite();
    if($dataSystem.optSideView) this.createHealthIconSprite();
};

Myth.BLK._SpriteEnemy_initMembers = Sprite_Enemy.prototype.initMembers;
Sprite_Enemy.prototype.initMembers = function() {
    Myth.BLK._SpriteEnemy_initMembers.call(this);    
    this.createBlockIconSprite();
    this.createHealthIconSprite();
};

Sprite_Battler.prototype.createBlockIconSprite = function ()
{
    this._blockIconSprites = [];

    for (var i = 0; i < Myth.BLK._blockStatesList.length; i++)
    {
        const bsList = Myth.BLK._blockStatesList[i];
        if (bsList)
        {
            this._blockIconSprites.push(new Sprite_BlockIcon(bsList.blockStateID));
        }
    }

    for (var j = 0; j < this._blockIconSprites.length; j++)
    {
        this.addChild(this._blockIconSprites[j]);
    }
};

Sprite_Battler.prototype.createHealthIconSprite = function ()
{
    this._rescIconSprites = [];

    for (var i = 0; i < Myth.BLK._rescStatesList.length; i++)
    {
        const bsList = Myth.BLK._rescStatesList[i];
        if (bsList)
        {
            this._rescIconSprites.push(new Sprite_HealthIcon(bsList.rescStateID));
        }
    }

    for (var j = 0; j < this._rescIconSprites.length; j++)
    {
        this.addChild(this._rescIconSprites[j]);
    }
}

Myth.BLK._SpriteBattler_setBattler = Sprite_Battler.prototype.setBattler;
Sprite_Battler.prototype.setBattler = function (battler)
{
    Myth.BLK._SpriteBattler_setBattler.call(this, battler);

    if (this._blockIconSprites)
    {
        for (var i = 0; i < this._blockIconSprites.length; i++)     this._blockIconSprites[i].setup(battler);
    }

    if (this._rescIconSprites)
    {
        for (var j = 0; j < this._rescIconSprites.length; j++)      this._rescIconSprites[j].setup(battler);
    }
};

//=============================================================================
// Game_Action
//=============================================================================

Myth.BLK._GameAction_executeDamage = Game_Action.prototype.executeDamage;
Game_Action.prototype.executeDamage = function(target, value)
{
    var skipBlock = false;

    var blockStates = target._states.filter(a => Myth.BLK.isBlockStateValid(a));

    //Damage Types/Element Filter
    blockStates = blockStates.filter((a) =>
    {
        if (Myth.BLK.getBlockStateInfo(a).blockedTypes.includes(this.item().damage.elementId)) return true;
        return false;
    });

    //Ignore Block Filter
    blockStates = blockStates.filter((a) =>
    {
        var dataArray = (this._item._dataClass == "skill") ? $dataSkills : $dataItems;
        if (dataArray[this._item._itemId]._ignoreBlock[a]) return false;
        return true;
    });

    //If at this point there are no valid Block States, skip and just resolve Damage as usual
    if (blockStates.length == 0) skipBlock = true;

    if (this.isHpEffect() && value > 0 && !skipBlock)
    {
        switch (Myth.BLK.multiBlockRule)
        {
            case "Ascending": blockStates = blockStates.sort(function (a, b) { return a - b }); break;
            case "Descending": blockStates = blockStates.sort(function (a, b) { return b - a }); break;
            case "Even Spread": blockStates = blockStates.filter(a => target._blockCount[a] > 0); break;
        }

        if (blockStates.length > 0)
        {
            if (Myth.BLK.multiBlockRule != "Even Spread")
            {
                for (var i = 0; i < blockStates.length; i++)
                {
                    if (target.getBlock(blockStates[i]) > value)
                    {
                        if (Myth.BLK.getBlockStateInfo(blockStates[i]).allowDamagePopup) target.startBlockPopup(blockStates[i], value);
                        target._blockCount[blockStates[i]] -= value;
                        target._lastBlocked = { id: blockStates[i], amt: value };
                        value = 0;
                    }
                    else if (target.getBlock(blockStates[i]) <= value)
                    {
                        if (Myth.BLK.getBlockStateInfo(blockStates[i]).allowDamagePopup) target.startBlockPopup(blockStates[i], target._blockCount[blockStates[i]]);
                        value -= target._blockCount[blockStates[i]];
                        target._lastBlocked = { id: blockStates[i], amt: target._blockCount[blockStates[i]] };
                        target._blockCount[blockStates[i]] = 0;
                        target.clearBlock(blockStates[i], true);
                    }

                    if (value == 0) break;
                }
            }
            else
            {
                var totalBlock = 0;
                for (var i = 0; i < blockStates.length; i++) totalBlock += target.getBlock(blockStates[i]);

                if (totalBlock > value)
                {
                    var valueAmt = Math.floor(value / blockStates.length);

                    for (var i = 0; i < blockStates.length; i++)
                    {
                        if (Myth.BLK.getBlockStateInfo(blockStates[i]).allowDamagePopup) target.startBlockPopup(blockStates[i], valueAmt);
                        target._blockCount[blockStates[i]] -= valueAmt;
                        target._lastBlocked = { id: blockStates[i], amt: valueAmt };
                    }

                    value = 0;
                }
                else if(totalBlock <= value)
                {
                    for (var i = 0; i < blockStates.length; i++)
                    {
                        if (Myth.BLK.getBlockStateInfo(blockStates[i]).allowDamagePopup) target.startBlockPopup(blockStates[i], target._blockCount[blockStates[i]]);
                        value -= target._blockCount[blockStates[i]];
                        target._lastBlocked = { id: blockStates[i], amt: target._blockCount[blockStates[i]] };
                        target._blockCount[blockStates[i]] = 0;
                        target.clearBlock(blockStates[i], true);
                    }
                }

            }
        }
    }

    Myth.BLK.windowRefresh();

	Myth.BLK._GameAction_executeDamage.call(this, target, value);
}

Myth.BLK._GameAction_itemEffectAddState = Game_Action.prototype.itemEffectAddState;
Game_Action.prototype.itemEffectAddState = function(target, effect)
{
    //if(Myth.BLK._negateStateIds.includes(effect.dataId)) target.clearBlock();
    Myth.BLK._GameAction_itemEffectAddState.call(this, target, effect);
}

//=============================================================================
// Window_BattleActor
//=============================================================================

Myth.BLK._Window_BattleActor_drawActorIcons = Window_BattleActor.prototype.drawActorIcons;
Window_BattleActor.prototype.drawActorIcons = function(actor, x, y, width) {

    var iconArray = actor.allIcons();
    var iconWidth = (Myth.Util.usingMZ ? ImageManager.iconWidth : Window_Base._iconWidth);
    var iconHeight = (Myth.Util.usingMZ ? ImageManager.iconHeight : Window_Base._iconHeight);

    for (var k = 0; k < Myth.BLK._blockStatesList.length; k++)
    {
        const bsList = Myth.BLK._blockStatesList[k];
        if (bsList)
        {
            var stateID = bsList.blockStateID;

            if (actor.isStateAffected(stateID) && $dataActors[actor._actorId]._hideBlock[stateID])
            {
                iconArray = iconArray.filter((item) => { return item != $dataStates[stateID].iconIndex });
            }
        }
    }

    for (var j = 0; j < Myth.BLK._rescStatesList.length; j++)
    {
        const rsList = Myth.BLK._rescStatesList[j];
        if (rsList)
        {
            var stateID = rsList.rescStateID;

            if (actor.isStateAffected(stateID) && $dataActors[actor._actorId]._hideResc[stateID])
            {
                iconArray = iconArray.filter((item) => { return item != $dataStates[stateID].iconIndex });
            }
        }
    }

    var icons = iconArray.slice(0, Math.floor(width / iconWidth));

    for (var i = 0; i < icons.length; i++)
    {
        this.drawIcon(icons[i], x + (iconWidth * i), y + 2);

        for (var j = 0; j < Myth.BLK._blockStatesList.length; j++)
        {
            const bsList = Myth.BLK._blockStatesList[j];
            if (bsList)
            {
                var stateID = bsList.blockStateID;

                if (icons[i] == $dataStates[stateID].iconIndex && actor.isStateAffected(stateID))
                {
                    this.contents.fontSize = bsList.blockIcon.iconFontSize;
                    this.contents.fontFace = bsList.blockIcon.iconFontFace;
                    this.drawText(String(actor.getBlock(stateID)),
                        x + (iconWidth * i) + ((3 - String(actor.getBlock(stateID)).length) * 4) + 2,
                        y + 2, this.textWidth("000"), iconHeight, 'center');
                    this.resetFontSettings();
                }
            }
        }

        for (var k = 0; k < Myth.BLK._rescStatesList.length; k++)
        {
            const rsList = Myth.BLK._rescStatesList[k];
            if (rsList)
            {
                var stateID = rsList.rescStateID;

                if (icons[i] == $dataStates[stateID].iconIndex && actor.isStateAffected(stateID))
                {
                    this.contents.fontSize = rsList.rescIcon.iconFontSize;
                    this.contents.fontFace = rsList.rescIcon.iconFontFace;

                    var evalValue = eval("actor." + String(Myth.BLK._rescStatesList[k].rescType).toLowerCase());

                    this.drawText(String(evalValue), x + (iconWidth * i) + ((3 - String(evalValue).length) * 4) + 2,
                        y + 2, this.textWidth("000"), iconHeight, 'center');
                    this.resetFontSettings();
                }
            }
        }
    }
};

//=============================================================================
// Window_PartyStatus
//=============================================================================

if (Myth.CGC.PartyUI)
{
    Myth.BLK._Window_PartyStatus_drawActorIcons = Window_PartyStatus.prototype.drawActorIcons;
    Window_PartyStatus.prototype.drawActorIcons = function (actor, x, y, width)
    {

        var iconArray = actor.allIcons();
        var iconWidth = (Myth.Util.usingMZ ? ImageManager.iconWidth : Window_Base._iconWidth);
        var iconHeight = (Myth.Util.usingMZ ? ImageManager.iconHeight : Window_Base._iconHeight);

        for (var k = 0; k < Myth.BLK._blockStatesList.length; k++)
        {
            const bsList = Myth.BLK._blockStatesList[k];
            if (bsList)
            {
                var stateID = bsList.blockStateID;

                if (actor.isStateAffected(stateID) && $dataActors[actor._actorId]._hideBlock[stateID]) {
                    iconArray = iconArray.filter((item) => { return item != $dataStates[stateID].iconIndex });
                }
            }
        }

        for (var j = 0; j < Myth.BLK._rescStatesList.length; j++)
        {
            const rsList = Myth.BLK._rescStatesList[j];
            if (rsList) {
                var stateID = rsList.rescStateID;

                if (actor.isStateAffected(stateID) && $dataActors[actor._actorId]._hideResc[stateID]) {
                    iconArray = iconArray.filter((item) => { return item != $dataStates[stateID].iconIndex });
                }
            }
        }

        var icons = iconArray.slice(0, Math.floor(width / iconWidth));

        for (var i = 0; i < icons.length; i++)
        {
            this.drawIcon(icons[i], x + (iconWidth * i), y + 2);

            for (var j = 0; j < Myth.BLK._blockStatesList.length; j++)
            {
                const bsList = Myth.BLK._blockStatesList[j];
                if (bsList)
                {
                    var stateID = bsList.blockStateID;

                    if (icons[i] == $dataStates[stateID].iconIndex && actor.isStateAffected(stateID)) {
                        this.contents.fontSize = bsList.blockIcon.iconFontSize;
                        this.contents.fontFace = bsList.blockIcon.iconFontFace;
                        this.drawText(String(actor.getBlock(stateID)),
                            x + (iconWidth * i) + ((3 - String(actor.getBlock(stateID)).length) * 4) + 2,
                            y + 2, this.textWidth("000"), iconHeight, 'center');
                        this.resetFontSettings();
                    }
                }
            }

            for (var k = 0; k < Myth.BLK._rescStatesList.length; k++)
            {
                const rsList = Myth.BLK._rescStatesList[k];
                if (rsList)
                {
                    var stateID = rsList.rescStateID;

                    if (icons[i] == $dataStates[stateID].iconIndex && actor.isStateAffected(stateID)) {
                        this.contents.fontSize = rsList.rescIcon.iconFontSize;
                        this.contents.fontFace = rsList.rescIcon.iconFontFace;

                        var evalValue = eval("actor." + String(Myth.BLK._rescStatesList[k].rescType).toLowerCase());

                        this.drawText(String(evalValue), x + (iconWidth * i) + ((3 - String(evalValue).length) * 4) + 2,
                            y + 2, this.textWidth("000"), iconHeight, 'center');
                        this.resetFontSettings();
                    }
                }
            }
        }
    };

    if (Myth.Util.usingMZ)
    {
        Myth.BLK.Window_PartyStatus_drawItemStatus = Window_PartyStatus.prototype.drawItemStatus;
        Window_PartyStatus.prototype.drawItemStatus = function (index)
        {
            Myth.BLK.Window_PartyStatus_drawItemStatus.call(this, index);

            const actor = this.actor(index);
            const rect = this.itemRectWithPadding(index);
            var iconWidth = ImageManager.iconWidth;
            var iconHeight = ImageManager.iconHeight;
            var offset = 1;

            for (var j = 0; j < Myth.BLK._blockStatesList.length; j++)
            {
                const bsList = Myth.BLK._blockStatesList[j];
                if (bsList)
                {
                    var stateID = bsList.blockStateID;

                    if (actor.isStateAffected(stateID) && !$dataActors[actor._actorId]._hideBlock[stateID])
                    {
                        this.contents.fontSize = bsList.blockIcon.iconFontSize;
                        this.contents.fontFace = bsList.blockIcon.iconFontFace;

                        this.drawIcon($dataStates[stateID].iconIndex, this.stateIconX(rect) - 9 + iconWidth * offset, this.stateIconY(rect) - (iconHeight * 0.5) - 2);

                        this.drawText(String(actor.getBlock(stateID)),
                            this.stateIconX(rect) + (iconWidth * offset) + ((3 - String(actor.getBlock(stateID)).length) * 4) - 6,
                            this.stateIconY(rect) - (iconHeight * 0.5) - 2, this.textWidth("000"), iconHeight, 'center');
                        this.resetFontSettings();
                        offset++;
                    }
                }
            }

            for (var k = 0; k < Myth.BLK._rescStatesList.length; k++)
            {
                const rsList = Myth.BLK._rescStatesList[k];
                if (rsList) {
                    var stateID = rsList.rescStateID;

                    if (actor.isStateAffected(stateID) && !$dataActors[actor._actorId]._hideResc[stateID])
                    {
                        this.contents.fontSize = rsList.rescIcon.iconFontSize;
                        this.contents.fontFace = rsList.rescIcon.iconFontFace;

                        var evalValue = eval("actor." + String(Myth.BLK._rescStatesList[k].rescType).toLowerCase());

                        this.drawIcon($dataStates[stateID].iconIndex, this.stateIconX(rect) - 9 + iconWidth * offset, this.stateIconY(rect) - (iconHeight * 0.5) - 2);
                        this.drawText(String(evalValue), this.stateIconX(rect) + (iconWidth * offset) + ((3 - String(evalValue).length) * 4) - 6,
                            this.stateIconY(rect) - (iconHeight * 0.5) - 2, this.textWidth("000"), iconHeight, 'center');
                        this.resetFontSettings();
                        offset++;
                    }
                }
            }
        }

        Window_PartyStatus.prototype.placeBasicGauges = function (actor, x, y)
        {
            var gaugeWidth = ($dataSystem.optDisplayTp) ? this.width * 0.15 : this.width * 0.2;
            var iconOffset = (ImageManager.iconWidth * 2) - 10;

            this.placeGauge(actor, "hp", x + iconOffset, y);
            this.placeGauge(actor, "mp", x + iconOffset + gaugeWidth + 2, y);
            if ($dataSystem.optDisplayTp) {
                this.placeGauge(actor, "tp", x + iconOffset + gaugeWidth * 2 + 2, y);
            }
        };
    }
}

//=============================================================================
// Window_ActorStatus
//=============================================================================

if (Myth.CGC.PartyUI)
{
    Myth.BLK._Window_ActorStatus_drawActorIcons = Window_ActorStatus.prototype.drawActorIcons;
    Window_ActorStatus.prototype.drawActorIcons = function (actor, x, y, width)
    {
        var iconArray = actor.allIcons();
        var iconWidth = (Myth.Util.usingMZ ? ImageManager.iconWidth : Window_Base._iconWidth);
        var iconHeight = (Myth.Util.usingMZ ? ImageManager.iconHeight : Window_Base._iconHeight);

        for (var k = 0; k < Myth.BLK._blockStatesList.length; k++)
        {
            const bsList = Myth.BLK._blockStatesList[k];
            if (bsList) {
                var stateID = bsList.blockStateID;

                if (actor.isStateAffected(stateID) && $dataActors[actor._actorId]._hideBlock[stateID]) {
                    iconArray = iconArray.filter((item) => { return item != $dataStates[stateID].iconIndex });
                }
            }
        }

        for (var j = 0; j < Myth.BLK._rescStatesList.length; j++)
        {
            const rsList = Myth.BLK._rescStatesList[j];
            if (rsList)
            {
                var stateID = rsList.rescStateID;

                if (actor.isStateAffected(stateID) && $dataActors[actor._actorId]._hideResc[stateID]) {
                    iconArray = iconArray.filter((item) => { return item != $dataStates[stateID].iconIndex });
                }
            }
        }

        var icons = iconArray.slice(0, Math.floor(width / iconWidth));

        for (var i = 0; i < icons.length; i++)
        {
            this.drawIcon(icons[i], x + (iconWidth * i), y + 2);

            for (var j = 0; j < Myth.BLK._blockStatesList.length; j++)
            {
                const bsList = Myth.BLK._blockStatesList[j];
                if (bsList) {
                    var stateID = bsList.blockStateID;

                    if (icons[i] == $dataStates[stateID].iconIndex && actor.isStateAffected(stateID)) {
                        this.contents.fontSize = bsList.blockIcon.iconFontSize;
                        this.contents.fontFace = bsList.blockIcon.iconFontFace;
                        this.drawText(String(actor.getBlock(stateID)),
                            x + (iconWidth * i) + ((3 - String(actor.getBlock(stateID)).length) * 4) + 2,
                            y + 2, this.textWidth("000"), iconHeight, 'center');
                        this.resetFontSettings();
                    }
                }
            }

            for (var k = 0; k < Myth.BLK._rescStatesList.length; k++)
            {
                const rsList = Myth.BLK._rescStatesList[k];
                if (rsList) {
                    var stateID = rsList.rescStateID;

                    if (icons[i] == $dataStates[stateID].iconIndex && actor.isStateAffected(stateID)) {
                        this.contents.fontSize = rsList.rescIcon.iconFontSize;
                        this.contents.fontFace = rsList.rescIcon.iconFontFace;

                        var evalValue = eval("actor." + String(Myth.BLK._rescStatesList[k].rescType).toLowerCase());

                        this.drawText(String(evalValue), x + (iconWidth * i) + ((3 - String(evalValue).length) * 4) + 2,
                            y + 2, this.textWidth("000"), iconHeight, 'center');
                        this.resetFontSettings();
                    }
                }
            }
        }
    };

    if (Myth.Util.usingMZ)
    {
        Myth.BLK.Window_ActorStatus_drawItemStatus = Window_ActorStatus.prototype.drawItemStatus;
        Window_ActorStatus.prototype.drawItemStatus = function (index)
        {
            Myth.BLK.Window_ActorStatus_drawItemStatus.call(this, index);

            const actor = this.actor(index);
            const rect = this.itemRectWithPadding(index);
            var iconWidth = ImageManager.iconWidth;
            var iconHeight = ImageManager.iconHeight;
            var offset = 1;

            for (var j = 0; j < Myth.BLK._blockStatesList.length; j++)
            {
                const bsList = Myth.BLK._blockStatesList[j];
                if (bsList)
                {
                    var stateID = bsList.blockStateID;

                    if (actor.isStateAffected(stateID) && !$dataActors[actor._actorId]._hideBlock[stateID])
                    {
                        this.contents.fontSize = bsList.blockIcon.iconFontSize;
                        this.contents.fontFace = bsList.blockIcon.iconFontFace;

                        this.drawIcon($dataStates[stateID].iconIndex,
                            iconWidth * (offset - 1),
                            this.lineHeight() * ($dataSystem.optDisplayTp ? 2.5 : 2) - iconHeight - 4);

                        this.drawText(String(actor.getBlock(stateID)),
                            (iconWidth * (offset - 1)) + (iconWidth * 0.5) - 4,
                            this.lineHeight() * ($dataSystem.optDisplayTp ? 1.5 : 1) - 2,
                            this.textWidth("000"), iconHeight, 'center');

                        this.resetFontSettings();
                        offset++;
                    }
                }
            }

            for (var k = 0; k < Myth.BLK._rescStatesList.length; k++)
            {
                const rsList = Myth.BLK._rescStatesList[k];
                if (rsList) {
                    var stateID = rsList.rescStateID;

                    if (actor.isStateAffected(stateID) && !$dataActors[actor._actorId]._hideResc[stateID])
                    {
                        this.contents.fontSize = rsList.rescIcon.iconFontSize;
                        this.contents.fontFace = rsList.rescIcon.iconFontFace;

                        var evalValue = eval("actor." + String(Myth.BLK._rescStatesList[k].rescType).toLowerCase());

                        this.drawIcon($dataStates[stateID].iconIndex,
                            iconWidth * (offset - 1),
                            this.lineHeight() * ($dataSystem.optDisplayTp ? 2.5 : 2) - iconHeight - 4);

                        this.drawText(String(evalValue),
                            (iconWidth * (offset - 1)) + ((3 - String(evalValue).length) * 4),
                            this.lineHeight() * ($dataSystem.optDisplayTp ? 1.5 : 1) - 2,
                            this.textWidth("000"), iconHeight, 'center');

                        this.resetFontSettings();
                        offset++;
                    }
                }
            }
        }
    }
}

//=============================================================================
// Window_BattleLog
//=============================================================================

Window_BattleLog.prototype.displayAddedStates = function(target) {
    target.result().addedStateObjects().forEach(function(state) {
        var stateMsg = target.isActor() ? state.message1 : state.message2;
        if (state.id === target.deathStateId()) {
            this.push('performCollapse', target);
        }
        if (Myth.BLK.isBlockStateValid(state.id))    stateMsg = null;
        if (stateMsg) {
            this.push('popBaseLine');
            this.push('pushBaseLine');
            this.push('addText', target.name() + stateMsg);
            this.push('waitForEffect');
        }
    }, this);
};

Myth.BLK._BattleLog_makeHpDamageText = Window_BattleLog.prototype.makeHpDamageText;
Window_BattleLog.prototype.makeHpDamageText = function(target)
{
    var result = target.result();
    var damage = result.hpDamage;
    var isActor = target.isActor();
    var fmt;

    if (damage > 0 && result.drain)
    {
        fmt = isActor ? TextManager.actorDrain : TextManager.enemyDrain;
        return fmt.format(target.name(), TextManager.hp, damage);
    }
    else if (damage > 0)
    {
        if(target._lastBlocked.amt > 0)
        {
            fmt = Myth.BLK.getBlockStateInfo(target.id).blockMsgs.breakText;
        }
        else
        {
            fmt = isActor ? TextManager.actorDamage : TextManager.enemyDamage;
        }

        return fmt.format(target.name(), damage);
    }
    else if (damage < 0)
    {
        fmt = isActor ? TextManager.actorRecovery : TextManager.enemyRecovery;
        return fmt.format(target.name(), TextManager.hp, -damage);
    } 
    else 
    {
        if(target._lastBlocked.amt > 0)
        {
            damage = target._lastBlocked.amt;
            fmt = Myth.BLK.getBlockStateInfo(target.id).blockMsgs.blockText;
            return fmt.format(target.name(), damage);
        }
        else
        {
            fmt = isActor ? TextManager.actorNoDamage : TextManager.enemyNoDamage;
            return fmt.format(target.name());
        }
    }
}

//=============================================================================
// New Card Actions
//=============================================================================

Myth.BLK.Game_Actor_performCardAction = Game_Battler.prototype.performCardAction;

Game_Battler.prototype.performCardAction = function (action) {
    var originalAction = action;
    action = this.formatCardAction(action);

    if (action.match(/(?:Add Block )([\+\-]\d+)/i))
    {
        if (Myth.BLK._blockStatesList.length > 0)
        {
            this.addBlock(Myth.BLK._blockStatesList[0].blockStateID, parseInt(RegExp.$1), true);
        }
        else
        {
            Myth.BLK.warnMessage(0);
        }
    }
    else if (action.match(/(?:Add Block )(\d+)[ ]([\+\-]\d+)/i))
    {
        if (Myth.BLK._blockStatesList.length > 0)
        {
            if (this.isBlockStateValid(parseInt(RegExp.$1)))
            {
                this.addBlock(parseInt(RegExp.$1), parseInt(RegExp.$2), true);
            }
            else
            {
                Myth.BLK.warnMessage(0);
            }
        }
        else
        {
            Myth.BLK.warnMessage(0);
        }
    }
    else if (action.match(/(?:Add Block[ ](.*)[ ])([\+\-]\d+)/i))
    {
        if (Myth.BLK._blockStatesList.length > 0)
        {
            var id = Myth.Util.getStateID(String(RegExp.$1));

            if (this.isBlockStateValid(id))
            {
                this.addBlock(id, parseInt(RegExp.$2), true);
            }
            else
            {
                Myth.BLK.warnMessage(0);
            }
        }
        else
        {
            Myth.BLK.warnMessage(0);
        }
    }
    else if (action.match(/(?:Set Block )(\d+)[ ](\d+)/i))
    {
        if (Myth.BLK._blockStatesList.length > 0)
        {
            if (this.isBlockStateValid(parseInt(RegExp.$1)))
            {
                this.setBlock(parseInt(RegExp.$1), parseInt(RegExp.$2), true);
            }
            else
            {
                Myth.BLK.warnMessage(0);
            }
        }
        else
        {
            Myth.BLK.warnMessage(0);
        }
    }
    else if (action.match(/(?:Set Block )(\d+)/i))
    {
        if (Myth.BLK._blockStatesList.length > 0)
        {
            this.setBlock(Myth.BLK._blockStatesList[0].blockStateID, parseInt(RegExp.$1), true);
        }
        else
        {
            Myth.BLK.warnMessage(0);
        }
    }
    else if (action.match(/(?:Set Block[ ](.*)[ ])(\d+)/i))
    {
        if (Myth.BLK._blockStatesList.length > 0)
        {
            var id = Myth.Util.getStateID(String(RegExp.$1));

            if (this.isBlockStateValid(id))
            {
                this.setBlock(id, parseInt(RegExp.$2), true);
            }
            else
            {
                Myth.BLK.warnMessage(0);
            }
        }
        else
        {
            Myth.BLK.warnMessage(0);
        }
    }
    else if (action.match(/(?:Fill Block)[ ](\d+)/i))
    {
        if (Myth.BLK._blockStatesList.length > 0)
        {
            if (this.isBlockStateValid(parseInt(RegExp.$1)))
            {
                this.fillBlock(parseInt(RegExp.$1), true);
            }
        }
    }
    else if (action.match(/(?:Fill Block)[ ](.*)/i))
    {
        if (Myth.BLK._blockStatesList.length > 0)
        {
            var id = Myth.Util.getStateID(String(RegExp.$1));

            if (this.isBlockStateValid(id))
            {
                this.fillBlock(id, true);
            }
        }
    }
    else if (action.match(/(?:Fill Block)/i))
    {
        if (Myth.BLK._blockStatesList.length > 0)
        {
            this.fillBlock(Myth.BLK._blockStatesList[0].blockStateID, true);
        }
    }
    else if (action.match(/(?:Clear Block)[ ](\d+)/i))
    {
        if (Myth.BLK._blockStatesList.length > 0)
        {
            if (this.isBlockStateValid(parseInt(RegExp.$1)))
            {
                this.clearBlock(parseInt(RegExp.$1), true);
            }
        }
    }
    else if (action.match(/(?:Clear Block)[ ](.*)/i))
    {
        if (Myth.BLK._blockStatesList.length > 0)
        {
            var id = Myth.Util.getStateID(String(RegExp.$1));

            if (this.isBlockStateValid(id))
            {
                this.clearBlock(id, true);
            }
        }
    }
    else if (action.match(/(?:Clear Block)/i))
    {
        if (Myth.BLK._blockStatesList.length > 0)
        {
            this.clearBlock(Myth.BLK._blockStatesList[0].blockStateID, true);
        }
    }
    else
        Myth.BLK.Game_Actor_performCardAction.call(this, originalAction);
}

//=============================================================================
// DataManager
//=============================================================================

var Myth_BlockGenerate_loaded = false;

Myth.BLK.DataManager_isDatabaseLoaded = DataManager.isDatabaseLoaded;
DataManager.isDatabaseLoaded = function()
{
	if(!Myth.BLK.DataManager_isDatabaseLoaded.call(this)) return false;
	if(!Myth_BlockGenerate_loaded)
    {
        DataManager.processInitBlockNotetags($dataActors);
        DataManager.processInitBlockNotetags($dataEnemies);
        DataManager.processMaxBlockNotetags($dataActors);
        DataManager.processMaxBlockNotetags($dataEnemies);

        DataManager.processModInitBlockNotetags($dataWeapons);
        DataManager.processModInitBlockNotetags($dataArmors);
        DataManager.processModMaxBlockNotetags($dataWeapons);
        DataManager.processModMaxBlockNotetags($dataArmors);

        DataManager.processChangeBlockNotetags($dataStates);
        DataManager.processFillBlockNotetags($dataStates);
        DataManager.processClearBlockNotetags($dataStates);
        DataManager.processNegateBlockNotetags($dataStates);

        DataManager.processHideRescNotetags($dataActors);
        DataManager.processHideRescNotetags($dataEnemies);

        DataManager.processHideBlockNotetags($dataActors);
        DataManager.processHideBlockNotetags($dataEnemies);

        DataManager.processIgnoreBlockNotetags($dataSkills);
        DataManager.processIgnoreBlockNotetags($dataItems);

        Myth_BlockGenerate_loaded = true;
	}

	return true;
}

DataManager.processHideBlockNotetags = function(group)
{
	for(var n = 1; n < group.length; n++)
	{
    	var obj = group[n];
        var notedata = obj.note.split(/[\r\n]+/);

        obj._hideBlock = [];

        for(var i = 0; i < notedata.length; ++i)
		{
            var line = notedata[i];

            if (line.match(/<HIDE BLOCK(?: ?(.*))>/i))
            {
                if (Myth.BLK._blockStatesList.length > 0)
                {
                    var id = RegExp.$1;

                    if (id != parseInt(id))
                    {
                        var defaultId = 1;
                        if (Myth.BLK._blockStatesList[0]) defaultId = Myth.BLK._blockStatesList[0].blockStateID;

                        id = (String(id).length > 0) ? Myth.Util.getStateID(String(id)) : defaultId;
                    }

                    id = parseInt(id);
                    if (Myth.BLK.isBlockStateValid(id)) obj._hideBlock[id] = true;
                }
                else
                {
                    //Warning Message
                    //Myth.BLK.warnMessage(0);
                }
            }

            if (line.match(/<HIDE ALL BLOCK>/i))
            {
                if (Myth.BLK._blockStatesList.length > 0)
                {
                    Myth.BLK._blockStatesList.forEach((block) => { obj._hideBlock[block.blockStateID] = true; });
                }
                else
                {
                    //Warning Message
                    //Myth.BLK.warnMessage(0);
                }
            }
		}
    }
}

DataManager.processHideRescNotetags = function(group)
{
	for(var n = 1; n < group.length; n++)
	{
    	var obj = group[n];
		var notedata = obj.note.split(/[\r\n]+/);

        obj._hideResc = [];

        for (var i = 0; i < notedata.length; ++i)
        {
            var line = notedata[i];

            if (line.match(/<HIDE RESC(?: ?(.*))>/i))
            {
                if (Myth.BLK._rescStatesList.length > 0)
                {
                    var id = RegExp.$1;

                    if (id != parseInt(id))
                    {
                        var defaultId = 1;
                        if (Myth.BLK._rescStatesList[0]) defaultId = Myth.BLK._rescStatesList[0].rescStateID;

                        id = (String(id).length > 0) ? Myth.Util.getStateID(String(id)) : defaultId;
                    }

                    id = parseInt(id);
                    if (Myth.BLK.isRescStateValid(id)) obj._hideResc[id] = true;
                }
                else
                {
                    //Warning Message
                    //Myth.BLK.warnMessage(0);
                }
            }

            if (line.match(/<HIDE ALL RESC>/i))
            {
                if (Myth.BLK._rescStatesList.length > 0)
                {
                    Myth.BLK._rescStatesList.forEach((resc) => { obj._hideResc[resc.rescStateID] = true; });
                }
                else
                {
                    //Warning Message
                    //Myth.BLK.warnMessage(0);
                }
            }

        }

    }
}

DataManager.processInitBlockNotetags = function(group)
{
	for(var n = 1; n < group.length; n++)
	{
		var obj = group[n];
		var notedata = obj.note.split(/[\r\n]+/);

        obj._initBlock = [];

		for(var i = 0; i < notedata.length; ++i)
		{
            var line = notedata[i];

            if (line.match(/<INIT BLOCK(?: ?(.*)): ?(.*)>/i))
            {
                var id = RegExp.$1;
                var value = eval(RegExp.$2);

                if (id != parseInt(id))
                {
                    var defaultId = 1;
                    if (Myth.BLK._blockStatesList[0]) defaultId = Myth.BLK._blockStatesList[0].blockStateID;

                    id = (String(id).length > 0) ? Myth.Util.getStateID(String(id)) : defaultId;
                }

                id = parseInt(id);
                value = parseInt(value);
                if (value && Myth.BLK.isBlockStateValid(id))   obj._initBlock[id] = value;
            }
            else
            {
                //Warning Message
                //Myth.BLK.warnMessage(0);
            }
		}
    }
}

DataManager.processMaxBlockNotetags = function(group)
{
	for(var n = 1; n < group.length; n++)
	{
		var obj = group[n];
		var notedata = obj.note.split(/[\r\n]+/);

        obj._maxBlock = [];

		for(var i = 0; i < notedata.length; ++i)
		{
            var line = notedata[i];

            if (line.match(/<MAX BLOCK(?: ?(.*)): ?(.*)>/i))
            {
                var id = RegExp.$1;
                var value = eval(RegExp.$2);

                if (id != parseInt(id))
                {
                    var defaultId = 1;
                    if (Myth.BLK._blockStatesList[0]) defaultId = Myth.BLK._blockStatesList[0].blockStateID;

                    id = (String(id).length > 0) ? Myth.Util.getStateID(String(id)) : defaultId;
                }

                id = parseInt(id);
                value = parseInt(value);
                if (value && Myth.BLK.isBlockStateValid(id)) obj._maxBlock[id] = value;
            }
            else
            {
                //Warning Message
                //Myth.BLK.warnMessage(0);
            }
		}
	}
}

DataManager.processModInitBlockNotetags = function(group)
{
	for(var n = 1; n < group.length; n++)
    {
        const initMods = [];

		var obj = group[n];
		var notedata = obj.note.split(/[\r\n]+/);

		for(var i = 0; i < notedata.length; ++i)
		{
            var line = notedata[i];

            if (line.match(/<MOD INIT BLOCK(?: ?(.*)): ?([\+\-].*)>/i))
            {
                var id = RegExp.$1;
                var value = eval(RegExp.$2);

                if (id != parseInt(id))
                {
                    var defaultId = 1;
                    if (Myth.BLK._blockStatesList[0]) defaultId = Myth.BLK._blockStatesList[0].blockStateID;

                    id = (String(id).length > 0) ? Myth.Util.getStateID(String(id)) : defaultId;
                }

                id = parseInt(id);
                value = parseInt(value);
                if (value && Myth.BLK.isBlockStateValid(id))
                {
                    if (!initMods[id]) initMods[id] = 0;
                    initMods[id] += value;
                }
            }
            else
            {
                //Warning Message
                //Myth.BLK.warnMessage(0);
            }
        }

        obj._initMods = initMods;
	}
}

DataManager.processModMaxBlockNotetags = function(group)
{
	for(var n = 1; n < group.length; n++)
	{
        const maxMods = [];

		var obj = group[n];
		var notedata = obj.note.split(/[\r\n]+/);

		for(var i = 0; i < notedata.length; ++i)
        {
            var line = notedata[i];

            if (line.match(/<MOD MAX BLOCK(?: ?(.*)): ?([\+\-].*)>/i))
            {
                var id = RegExp.$1;
                var value = eval(RegExp.$2);

                if (id != parseInt(id))
                {
                    var defaultId = 1;
                    if (Myth.BLK._blockStatesList[0]) defaultId = Myth.BLK._blockStatesList[0].blockStateID;

                    id = (String(id).length > 0) ? Myth.Util.getStateID(String(id)) : defaultId;
                }

                id = parseInt(id);
                value = parseInt(value);
                if (value && Myth.BLK.isBlockStateValid(id))
                {
                    if (!maxMods[id]) maxMods[id] = 0;
                    maxMods[id] += value;
                }
            }
            else
            {
                //Warning Message
                //Myth.BLK.warnMessage(0);
            }
		}

        obj._maxMods = maxMods;
	}
}

DataManager.processChangeBlockNotetags = function(group)
{
	for(var n = 1; n < group.length; n++)
	{
		const blockChanges = [];

		var obj = group[n];
		var notedata = obj.note.split(/[\r\n]+/);

		for(var i = 0; i < notedata.length; ++i)
		{
            var line = notedata[i];

            if (line.match(/<CHANGE BLOCK(?: ?(.*)): ?([\+\-].*)>/i))
            {
                var id = RegExp.$1;
                var value = eval(RegExp.$2);

                if (id != parseInt(id))
                {
                    var defaultId = 1;
                    if (Myth.BLK._blockStatesList[0]) defaultId = Myth.BLK._blockStatesList[0].blockStateID;

                    id = (String(id).length > 0) ? Myth.Util.getStateID(String(id)) : defaultId;
                }

                id = parseInt(id);
                value = parseInt(value);

                if (value && Myth.BLK.isBlockStateValid(id))
                {
                    if (!blockChanges[id]) blockChanges[id] = 0;
                    blockChanges[id] += value;
                }
                else
                {
                    //Warning Message
                    //Myth.BLK.warnMessage(0);
                }
            }
        }

        obj._blockChanges = blockChanges;
	}
}

DataManager.processIgnoreBlockNotetags = function(group)
{
	for(var n = 1; n < group.length; n++)
    {
    	var obj = group[n];
		var notedata = obj.note.split(/[\r\n]+/);

        obj._ignoreBlock = [];

        for(var i = 0; i < notedata.length; ++i)
		{
            var line = notedata[i];

            if (line.match(/<IGNORE BLOCK(?: ?(.*))>/i))
            {
                var id = RegExp.$1;

                if (id != parseInt(id))
                {
                    var defaultId = 1;
                    if (Myth.BLK._blockStatesList[0]) defaultId = Myth.BLK._blockStatesList[0].blockStateID;

                    id = (String(id).length > 0) ? Myth.Util.getStateID(String(id)) : defaultId;
                }

                id = parseInt(id);
                if (Myth.BLK.isBlockStateValid(id))    obj._ignoreBlock[id] = true;
            }
            else
            {
                //Warning Message
                //Myth.BLK.warnMessage(0);
            }
        }
    }
}

DataManager.processFillBlockNotetags = function (group)
{
    for (var n = 1; n < group.length; n++)
    {
        var obj = group[n];
        var notedata = obj.note.split(/[\r\n]+/);

        obj._fillBlock = [];

        for (var i = 0; i < notedata.length; ++i)
        {
            var line = notedata[i];

            if (line.match(/<FILL BLOCK(?: ?(.*))>/i))
            {
                var id = RegExp.$1;

                if (id != parseInt(id))
                {
                    var defaultId = 1;
                    if (Myth.BLK._blockStatesList[0]) defaultId = Myth.BLK._blockStatesList[0].blockStateID;

                    id = (String(id).length > 0) ? Myth.Util.getStateID(String(id)) : defaultId;
                }

                id = parseInt(id);
                if (Myth.BLK.isBlockStateValid(id)) obj._fillBlock[id] = true;
            }
            else
            {
                //Warning Message
                //Myth.BLK.warnMessage(0);
            }
        }
    }
}

DataManager.processClearBlockNotetags = function (group)
{
    for (var n = 1; n < group.length; n++)
    {
        var obj = group[n];
        var notedata = obj.note.split(/[\r\n]+/);

        obj._clearBlock = [];

        for (var i = 0; i < notedata.length; ++i)
        {
            var line = notedata[i];

            if (line.match(/<CLEAR BLOCK(?: ?(.*))>/i))
            {
                var id = RegExp.$1;

                if (id != parseInt(id))
                {
                    var defaultId = 1;
                    if (Myth.BLK._blockStatesList[0]) defaultId = Myth.BLK._blockStatesList[0].blockStateID;

                    id = (String(id).length > 0) ? Myth.Util.getStateID(String(id)) : defaultId;
                }

                id = parseInt(id);
                if (Myth.BLK.isBlockStateValid(id)) obj._clearBlock[id] = true;
            }
            else
            {
                //Warning Message
                //Myth.BLK.warnMessage(0);
            }
        }
    }
}

DataManager.processNegateBlockNotetags = function (group)
{
    for (var n = 1; n < group.length; n++)
    {
        var obj = group[n];
        var notedata = obj.note.split(/[\r\n]+/);

        obj._negateBlock = [];

        for (var i = 0; i < notedata.length; ++i)
        {
            var line = notedata[i];

            if (line.match(/<NEGATE BLOCK(?: ?(.*))>/i))
            {
                var id = RegExp.$1;

                if (id != parseInt(id))
                {
                    var defaultId = 1;
                    if (Myth.BLK._blockStatesList[0]) defaultId = Myth.BLK._blockStatesList[0].blockStateID;

                    id = (String(id).length > 0) ? Myth.Util.getStateID(String(id)) : defaultId;
                }

                id = parseInt(id);
                if (Myth.BLK.isBlockStateValid(id)) obj._negateBlock[id] = true;
            }
            else
            {
                //Warning Message
                //Myth.BLK.warnMessage(0);
            }
        }
    }
}
